//
//	GSXSerialCommunicator.h
//	 1998, 1999, 2000 Kyle Hammond
//	hammo009@tc.umn.edu
//	Use at your own risk.  All rights reserved.  Do not distribute without permission.
//

/*	Use these classes something like this:

{	GSXSerialCommunicator			*serial = nil;
	GSXSerialPortNameRecordHandle	portListH = nil;
	GSXSerialPortNameRecord		portName;
	GSSerialPortInfoRecord			portInfo = { 9600, 8, kGSSerialParityNone, kGSSerialStopBits10 };

#if defined( macintosh )
	if ( useOpenTransport ) {
		GSSerialOT::FindReadySerialPorts( &portListH, false if OpenTransport is already initialized );

		// Determine which port to use.  Probably allow the user to select it based on the userPortName.
		portName = (*portListH)[ user-selected index into the array ];

		serial = new GSSerialOT( &portInfo, &portName );
	} else {
		GSSerialClassic::FindReadySerialPorts( &portListH );

		// Determine which port to use.  Probably allow the user to select it based on the userPortName.
		portName = (*portListH)[ user-selected index into the array ];

		serial = new GSSerialClassic( &portInfo, &portName, false for synchronous );
	}
#else
	GSSerialWin32::FindReadySerialPorts( &portListH );

	// Determine which port to use.  Probably allow the user to select it based on the userPortName.
	portName = (*portListH)[ user-selected index into the array ];

	serial = new GSSerialWin32( &portInfo, &portName, true for asynchronous );
#endif

	serial->InitSerialComm( );

	// Use the SendSerialInfo and GetSerialInfo functions in here.

	serial->CloseSerialComm( );

	delete serial;
}



	General Notes:
		a. Be sure to InitSerialComm( ) before trying to send or receive data.
		b. Be sure to CloseSerialComm( ) when finished with the serial port.
		c. Be sure that each hardware serial port has only one instance of a GSXSerialCommunicator associated with it.
			For instance, under Windows 95, you could do something like:
				GSXSerialPortNameRecordHandle	portListH = nil;
				GSSerialWin32			*serial1, *serial2, *serial3;
				GSSerialPortInfoRecord	genericInfo;

				genericInfo.baud = 9600;	genericInfo.parity = kGSSerialParityNone;
				genericInfo.dataBits = 8;	genericInfo.stopBits = kGSSerialStopBits10;

				GSSerialWin32::FindReadySerialPorts( &portListH );

				// These three lines assume that there are at least three ports available.
				serial1 = new GSSerialWin32( &genericInfo, *portList, false );
				serial2 = new GSSerialWin32( &genericInfo, &(*portList)[ 1 ], false );
				serial3 = new GSSerialWin32( &genericInfo, &(*portList)[ 2 ], false );
		d. FindReadySerialPorts looks for all ports and returns a handle (a pointer to a pointer under Win32) that contains a
				variable size array of ports, with user names (Modem Port) and internal names ( .Ain, .Aout ).
				FindReadySerialPorts also returns the number of ports found.


	MacOS Notes:
		Important Note: The GSSerialMacOS.cpp code is now based on version 3.3 of Apple's Universal Headers.
				You can download the headers for free from Apple's developer web site.

		a. The Open Transport implementation is preferred; Open Transport is newer technology.
			And it's not Carbon compatible (as of November 24, 1999)!
			a1.  Oops!  Open Transport will not support serial endpoints under Carbon.  So, we'll all have to wait
				until Apple releases the IOKit so we can see how RS-232 serial communications will work.

		b. The incoming data buffer is only 256 bytes per port.  If you are reading large chunks of data from the serial port, you
				should probably increase that, or use multiple buffers.  Search for kGSSerialMacBufferSize for details.

		c. All return values are actually OSErr; the function prototypes use unsigned long for compatibility with Win32.

		d. In asynchronous mode, Get/SendSerialInfo return values only indicate if the initial command to read/write is succesful;
				the return values do not reflect if any data was actually sent or recieved.

		OT Notes:
			a. The OpenTransport code is always asynchronous.  I don't think it's possible to use synchronous calls.
			b. The header file OpenTptConfig.h does not come with the standard Codewarrior headers.  It is available in the
					OpenTransport SDK from the Apple web site at http://developer.apple.com/sdk/index.html.
				b2. The required header file is included in Universal Headers v3.3.
			c. Under Carbon, there are a couple problems left:
				c1. Carbon OT will not support serial endpoints, so this isn't really Carbon compatible.
				c2. How do I correctly call OTYieldPortRequest when I can't call OTGetProviderPortRef for the serial endpoint?
				c3. How do I get user readable names for ports (OTGetUserPortNameFromPortRef is not defined in Carbon)?

		Classic Notes:
			a. Default mode is asynchronous ( see constructor specification ).
			b. Should always use the asynchronous mode because PBReadSync will block until enough data is read; this is bad if
						the Mac is not connected to anything (no data will EVER come in and your app will freeze the machine).
			c. The default baud rate is 9600 if the GSSerialPortInfoRecord baud field is not a supported rate.


	Win32 Notes:
		- The code for finding the available serial ports was originally written by Merle F. McClelland.
				Thanks for allowing me to include it with this distribution.

		a. Since adding the asynchronous code, I have not had a chance to test this.
		b. Currently, only the synchronous mode definitely works properly.

*/

#if !defined( __GSXSerialCommunicator_h_ )
#define __GSXSerialCommunicator_h_	1

#if defined( macintosh )
	#if !defined( __MACTYPES__ )
		#include <MacTypes.h>
	#endif
	#if !defined( __STRINGCOMPARE__ )
		#include <StringCompare.h>
	#endif
#endif

#if PRAGMA_STRUCT_ALIGN
	// This allows you to store the serial port record on 68K and PowerPC MacOS in the same format.
	#pragma options align=power
#endif

// A structure to contain the serial port configuration.
typedef struct {
	unsigned long	baud;				// Simply the baud rate in decimal.
	char			dataBits;				// The number of bits per byte to send (in decimal).
	char			parity;				// For parity and stopBits use the constants
	char			stopBits;				// 		enumerated below.
} GSSerialPortInfoRecord;

#if !defined( macintosh )
	const short kWin32PortNameLength = 64;
#endif

typedef struct {
#if defined( macintosh )
	Str63	userPortName;		// Display this in a list for the user to choose from.
	Str63	inPortName;		// Used internally.
	Str63	outPortName;		// Used internally.
#else
	char	userPortName[ kWin32PortNameLength ];	// Display this in a list for the user to choose from.
	char	inPortName[ kWin32PortNameLength ];	// Used internally.
	char	outPortName[ kWin32PortNameLength ];	// Used internally.
#endif
} GSXSerialPortNameRecord, **GSXSerialPortNameRecordHandle;

#if PRAGMA_STRUCT_ALIGN
	#pragma options align=reset
#endif

enum {
	// parity field values.
	kGSSerialParityNone = 0,
	kGSSerialParityOdd,
	kGSSerialParityEven,

	// stopBits field values.
	kGSSerialStopBits10 = 0,		// 1.0 stop bit
	kGSSerialStopBits15,		// 1.5 stop bits
	kGSSerialStopBits20			// 2.0 stop bits
};


class GSXSerialCommunicator {
	// Abstract Base Class for all serial communicators.
	// Currently, there are two MacOS implementations and one Win32 implementation.
public:
	// Initialize and close serial communications.
	virtual unsigned long	InitSerialComm( void ) = 0;
	virtual unsigned long	CloseSerialComm( void ) = 0;

	// Change the port settings (including switching ports).
	virtual unsigned long	SetPortOptions( const GSSerialPortInfoRecord *inPortInfo, const GSXSerialPortNameRecord *inPortName ) = 0;

	// Get current port settings.  Either parameter may be NULL if you are not interested in that info.
	virtual unsigned long	GetPortOptions( GSSerialPortInfoRecord *outPortInfo, GSXSerialPortNameRecord *outPortName );

	// Send and get data over the serial port.
	virtual unsigned long	SendSerialInfo( const void *inBuffer, const unsigned long inBufferSize ) = 0;
	virtual unsigned long	GetSerialInfo( void *outBuffer, unsigned long *ioCount, const long inWaitTicks ) = 0;

	// Destructor.
	virtual ~GSXSerialCommunicator( );

// Implementation available to sub-classes.
protected:
	GSSerialPortInfoRecord		mCurrentPortInfo;	// The current port settings
	GSXSerialPortNameRecord	mCurrentPortName;	// The current port name
	bool						mAsync;			// Are we in asynchronous mode?

	// Constructor.
	GSXSerialCommunicator( const GSSerialPortInfoRecord *inPortInfo, const GSXSerialPortNameRecord *inPortName,
				const bool inAsynchronous );

private:
	// Can't use the default constructor or the copy constructor.
	GSXSerialCommunicator( );
	GSXSerialCommunicator( const GSXSerialCommunicator & );
};

// Tests to see if inPort1 and inPort2 refer to the same port.  The user names are not checked or changed.
bool PortRecordsEqual( const GSXSerialPortNameRecord *inPort1, const GSXSerialPortNameRecord *inPort2 );


// Inline the simple functions.

inline unsigned long GSXSerialCommunicator::GetPortOptions( GSSerialPortInfoRecord *outPortInfo, GSXSerialPortNameRecord *outPortName )
{	if ( outPortInfo != NULL )	*outPortInfo = mCurrentPortInfo;
	if ( outPortName != NULL )	*outPortName = mCurrentPortName;
	return 0;
}

inline GSXSerialCommunicator::~GSXSerialCommunicator( )
{	}

inline GSXSerialCommunicator::GSXSerialCommunicator( const GSSerialPortInfoRecord *inPortInfo,
			const GSXSerialPortNameRecord *inPortName, const bool inAsynchronous ) : mCurrentPortInfo( *inPortInfo ),
			mCurrentPortName( *inPortName ), mAsync( inAsynchronous )
{	}

inline bool PortRecordsEqual( const GSXSerialPortNameRecord *inPort1, const GSXSerialPortNameRecord *inPort2 )
{	// Compare only the in and out name; don't compare the user name.
#if defined( macintosh )
	if ( EqualString( inPort1->inPortName, inPort2->inPortName, true, true ) )
		return EqualString( inPort1->outPortName, inPort2->outPortName, true, true );

	return false;
#else
	if ( strcmp( inPort1->inPortName, inPort2->inPortName ) == 0 )
		return ( strcmp( inPort1->outPortName, inPort2->outPortName ) == 0 );

	return false;
#endif
}

#endif